/*
 * cSimMsgQueue.cpp
 * 
 * Class implementation for a message queue.
 *
 */
#include "cSimMsgQueue.h"
#include "cSimInfo.h"
#include "Endpoint/cEndpoint.h"

#ifndef NULL
	#define NULL 0
#endif

/*
 * cSimMsgQueue::cSimMsgQueue()
 *
 * Purpose:	Creates a new msg queue.
 * IN:		initialCap	-> The starting size of the queue.
 *			capInc		-> How much to grow the queue by every time.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue is set up.
 * Throws:	-
 * Return:	-
 */
cSimMsgQueue::cSimMsgQueue(unsigned int initialCap, unsigned int capInc)
{
	mSize			= initialCap;
	mInsertPtr		= 0;
	mCapIncrement   = capInc;

	// Alloc queue space.
	mQueue = new SimMsg[mSize];
	if(!mQueue)
	{
		//throw cQueueException("<cSortedMsgQueue::cSortedMsgQueue>: Unable to alloc new queue.");
	}
	for(unsigned int i = 0; i < mSize; i++)
	{
		mQueue[i].destEp = NULL;
	}
}

/*
 * cSimMsgQueue::~cSimMsgQueue()
 *
 * Purpose:	Destroy the queue.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue is cleaned up.
 * Return:	-
 */
cSimMsgQueue::~cSimMsgQueue()
{
	if(mQueue) 
	{ 
		Clear();
		delete mQueue; 
	}
}

/*
 * cSimMsgQueue::_Delete()
 *
 * Purpose:	Deletes the element at the given index.
 * IN:		index		-> Index of the element to delete.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue shrinks by at least one.
 * Return:	true if success, else false.
 */
bool cSimMsgQueue::_Delete(unsigned int index)
{
	mQueue[index].destEp = NULL;

	// Swap elements over to fill hole
	index = index + 1;
	while(index < (int)mInsertPtr)
	{
		mQueue[index-1] = mQueue[index];
		index++;
	}
	mInsertPtr--;
	mQueue[mInsertPtr].destEp = NULL;
	return true;
}

/*
 * cSimMsgQueue::_Grow()
 *
 * Purpose:	Attempts to grow the queue by at least one, at most mCapIncrement
 * IN:		-
 * OUT:		-
 * Cond:	The queue is full!
 * PostCnd:	The queue is grown by at least one, at most mCapIncrement elements.
 * Return:	true if success, else false.
 */ 
bool cSimMsgQueue::_Grow()
{
	if(mInsertPtr == mSize)
	{
		// We need to expand the queue.
		SimMsg* newQ = new SimMsg[mSize+mCapIncrement];
		if(!newQ)
		{
			return false;
		}

		// The following may be faster using memcpy()...
		for(unsigned int i = 0; i < mSize; i++)
		{
			newQ[i] = mQueue[i];
		}
		for(i = mSize; i < mSize+mCapIncrement; i++)
		{
			newQ[i].destEp = NULL;
		}
		mSize = mSize + mCapIncrement;
		delete mQueue;
		mQueue = newQ;
	}
 	return true;
}

/*
 * cSimMsgQueue::Insert()
 *
 * Purpose:	Attempts to insert the message.
 * IN:		msg		-> The message to insert
 * OUT:		-
 * Cond:	-
 * PostCnd:	The msg is placed in the queue.
 * Return:	true if success, else false.
 */ 
bool cSimMsgQueue::Insert(SimMsg* msg)
{
	// Inserting brand new element!
	if(mInsertPtr == mSize)
	{
		if(!_Grow())
		{
			return false;
		}
	}
	mQueue[mInsertPtr] = *msg;

	// Sort it
	_InsertSort();
	mInsertPtr = mInsertPtr+1;	// Not a circular queue!

	return true;
}

/*
 * cSimMsgQueue::_InsertSort()
 *
 * Purpose:	Keeps the sorted property of the queue, assuming new element is placed
 *			at mInsertPtr
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if a should be delivered before b, else false.
 */ 
bool cSimMsgQueue::_InsertSort()
{
	unsigned int	current, prev;
	SimMsg			temp;

	current = prev = mInsertPtr;
	while(current != 0)
	{
		prev = prev-1;		// No wrap around!

		// Make sure that prev is not a hole!
		if(mQueue[prev].destEp)
		{
			if(!_Before(mQueue[current].destEp, mQueue[prev].destEp))
			{
				return true;		// No need to swap anymore
			}
		}

		// Swap (into a hole if one exists)
		temp = mQueue[current];
		mQueue[current] = mQueue[prev];
		mQueue[prev] = temp;

		current = prev;
	}
	return true;
}

/*
 * cSimMsgQueue::_Before()
 *
 * Purpose:	Returns true if a is before b
 * IN:		a, b	-> msgs to compare.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if a is before b, else false.
 */ 
bool cSimMsgQueue::_Before(cEndpoint* a, cEndpoint* b)
{
	return a->LessThan(b);
}

/*
 * cSimMsgQueue::_Equal()
 *
 * Purpose:	Returns true if a refers to same subnet as b
 * IN:		a, b	-> msgs to compare.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if a is the same as b
 */ 
bool cSimMsgQueue::_Equal(cEndpoint* a, cEndpoint* b)
{
	return a->Equals(b);
}

/*
 * cSimMsgQueue::Find()
 *
 * Purpose:	Searches for the element and returns its index.
 * IN:		ep		-> basis for search
 * OUT:		result	-> the node, if it is found.
 * Cond:	
 * PostCnd:	The found message is *removed* from the queue!
 * Return:	true if found, else false.
 */
bool cSimMsgQueue::Find(cEndpoint* ep, SimMsg* msg)
{
	int retVal;

	if(!mInsertPtr)
	{
		return false;
	}

	retVal = _BinSearch(ep, 0, mInsertPtr-1);
	if(retVal != -1)
	{
		*msg = mQueue[retVal];
		_Delete(retVal);	// Remove from queue.
	}
	return (retVal != -1);
}

/*
 * cSimMsgQueue::_BinSearch()
 *
 * Purpose:	Performs binary search, looking for a match.
 * NOTE:	The input left/right can NOT be wrap around values.
 * IN:		ep			-> The endpoint to look for.
 *			left		-> The left bounds of the array (start w/ 0)
 *			right		-> The right bounds of the array (start w/ insert ptr-1)
 * OUT:		-
 * Cond:	The endpoint is in the queue.
 * PostCnd:	-
 * Return:	-1 if not found, else the index in mQueue of the SubnetInfo
 */
int cSimMsgQueue::_BinSearch(cEndpoint* ep, int left, int right)
{
	int pivot;

	while(1)
	{
		if(left > right)
		{
			return -1;
		}
		else
		{
			pivot = (right-left) / 2;	// Just how much pivot advances from left, actually.
		}
		pivot = (left + pivot);	

		// See where to search next
		if(_Before(ep, mQueue[pivot].destEp))
		{
			right = pivot-1;
		}
		else if( _Equal(ep, mQueue[pivot].destEp) )
		{
			// Found it!
			return pivot;
		}
		else
		{
			// Search from pivot+1 to right
			left = pivot+1;
		}
	}
}

/*
 * cSimMsgQueue::Clear()
 *
 * Purpose:	Clears out the queue.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue is reset and cleared.
 * Return:	true if success, false otherwise.
 */
bool cSimMsgQueue::Clear()
{
	for(unsigned int i = 0; i < mSize; i++)
	{
		if(mQueue[i].destEp)
		{
			mQueue[i].destEp = NULL;
		}
	}
	mInsertPtr   = 0;
	return true;
}
